<?php
// database.php

class Database {
    private $pdo;

    public function __construct() {
        try {
            $dsn = "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME . ";charset=utf8mb4";
            $options = [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                PDO::ATTR_EMULATE_PREPARES => false,
            ];
            $this->pdo = new PDO($dsn, DB_USER, DB_PASS, $options);
            $this->initializeTables();
        } catch (PDOException $e) {
            error_log("Database Connection Error: " . $e->getMessage());
            exit("خطای اتصال به پایگاه داده.");
        }
    }

    // ایجاد خودکار جداول در صورت عدم وجود
    private function initializeTables() {
        // جدول کاربران
        $this->pdo->exec("CREATE TABLE IF NOT EXISTS users (
            id BIGINT PRIMARY KEY,
            username VARCHAR(255),
            display_name VARCHAR(255) DEFAULT NULL,
            step VARCHAR(50) DEFAULT 'none',
            unique_link VARCHAR(64) UNIQUE,
            is_active BOOLEAN DEFAULT TRUE,
            allowed_media JSON,
            temp_data JSON,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci");

        // جدول پیام‌ها (برای سیستم خواندن پیام و پاسخ)
        $this->pdo->exec("CREATE TABLE IF NOT EXISTS messages (
            id INT AUTO_INCREMENT PRIMARY KEY,
            sender_id BIGINT,
            sender_msg_id BIGINT,
            receiver_id BIGINT,
            message_type VARCHAR(20),
            file_id TEXT,
            caption TEXT,
            is_seen BOOLEAN DEFAULT FALSE,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci");

        // --- ویرایش: افزودن ستون برای ترک کردن پیام سمت گیرنده (جهت ریپلای دقیق) ---
        try {
            $this->pdo->exec("ALTER TABLE messages ADD COLUMN receiver_msg_id BIGINT DEFAULT NULL");
        } catch (Exception $e) {
            // ستون قبلاً وجود دارد
        }


        // جدول بلاک‌ها
        $this->pdo->exec("CREATE TABLE IF NOT EXISTS blocked_users (
            id INT AUTO_INCREMENT PRIMARY KEY,
            user_id BIGINT,
            blocked_id BIGINT,
            name VARCHAR(255),
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            UNIQUE(user_id, blocked_id)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci");

        // جدول مخاطبین
        $this->pdo->exec("CREATE TABLE IF NOT EXISTS contacts (
            id INT AUTO_INCREMENT PRIMARY KEY,
            user_id BIGINT,
            contact_id BIGINT,
            name VARCHAR(255),
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            UNIQUE(user_id, contact_id)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci");
        
        // جدول مدیران (برای مدیریت داینامیک)
              // جدول مدیران (برای مدیریت داینامیک)
        $this->pdo->exec("CREATE TABLE IF NOT EXISTS admins (
            user_id BIGINT PRIMARY KEY,
            name VARCHAR(255) DEFAULT 'ادمین',
            added_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci");
        
        // اطمینان از وجود ستون name برای دیتابیس‌های قدیمی
        try {
            $this->pdo->exec("ALTER TABLE admins ADD COLUMN name VARCHAR(255) DEFAULT 'ادمین'");
        } catch (Exception $e) { /* ستون وجود دارد، نادیده بگیر */ }

        // ایجاد ستون شماره موبایل در جدول کاربران اگر وجود نداشته باشد
        try {
            $this->pdo->exec("ALTER TABLE users ADD COLUMN phone_number VARCHAR(30) DEFAULT NULL");
        } catch (PDOException $e) {
            // ستون قبلاً وجود دارد
        }

        // افزودن ستون آخرین بازدید برای سیستم آماردهی
        try {
            $this->pdo->exec("ALTER TABLE users ADD COLUMN last_activity TIMESTAMP DEFAULT CURRENT_TIMESTAMP");
        } catch (PDOException $e) {
            // ستون قبلاً وجود دارد
        }


        // ایجاد جدول تنظیمات ربات
        $this->pdo->exec("CREATE TABLE IF NOT EXISTS bot_settings (
            setting_key VARCHAR(50) PRIMARY KEY, 
            setting_value VARCHAR(255)
        )");

        // مقداردهی اولیه تنظیمات اگر وجود نداشته باشند
        $check = $this->query("SELECT * FROM bot_settings WHERE setting_key = 'phone_lock_status'")->fetch();
        if (!$check) {
            $this->query("INSERT INTO bot_settings (setting_key, setting_value) VALUES ('phone_lock_status', 'off')");
            $this->query("INSERT INTO bot_settings (setting_key, setting_value) VALUES ('phone_lock_mode', 'iran')");
        }


        // --- افزودن جدول و تنظیمات عضویت اجباری ---
        
        // جدول لیست کانال‌ها و گروه‌های قفل اجباری
        $this->pdo->exec("CREATE TABLE IF NOT EXISTS force_join_channels (
            id INT AUTO_INCREMENT PRIMARY KEY,
            chat_id BIGINT NOT NULL,
            chat_name VARCHAR(255),
            invite_link TEXT,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci");

        // تنظیم پیش‌فرض وضعیت عضویت اجباری
        $checkFJ = $this->query("SELECT * FROM bot_settings WHERE setting_key = 'force_join_status'")->fetch();
        if (!$checkFJ) {
            $this->query("INSERT INTO bot_settings (setting_key, setting_value) VALUES ('force_join_status', 'off')");
        }
        
                // --- افزودن جدول وضعیت ارسال پیام همگانی (برادکست) ---
        $this->pdo->exec("CREATE TABLE IF NOT EXISTS broadcast_state (
            id INT PRIMARY KEY,
            status VARCHAR(20) DEFAULT 'idle', -- وضعیت: idle (آزاد), sending (در حال ارسال)
            msg_type VARCHAR(20) DEFAULT NULL, -- نوع پیام: text, photo, forward, etc
            file_id TEXT DEFAULT NULL, -- آیدی فایل یا متن پیام
            caption TEXT DEFAULT NULL,
            forward_chat_id BIGINT DEFAULT NULL, -- برای فوروارد: آیدی کانال مبدا
            forward_msg_id INT DEFAULT NULL, -- برای فوروارد: آیدی پیام مبدا
            batch_offset INT DEFAULT 0, -- تعداد ارسال شده تا این لحظه
            success_count INT DEFAULT 0, -- تعداد موفق
            fail_count INT DEFAULT 0, -- تعداد ناموفق
            start_time TIMESTAMP NULL, -- زمان شروع
            total_users INT DEFAULT 0 -- تعداد کل گیرندگان در زمان شروع
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci");

        // ایجاد سطر اولیه برای مدیریت وضعیت (اگر وجود نداشته باشد)
        $this->pdo->exec("INSERT IGNORE INTO broadcast_state (id, status) VALUES (1, 'idle')");

               // --- اطمینان از وجود ستون زمان شروع (فیکس برای دیتابیس‌های قدیمی) ---
        try {
            $this->pdo->exec("ALTER TABLE broadcast_state ADD COLUMN start_time DATETIME DEFAULT NULL");
        } catch (Exception $e) {
            // اگر ستون از قبل وجود داشته باشد، خطا را نادیده بگیر
        }
        
        // اطمینان از وجود ستون‌های شمارشگر برای دیتابیس‌های قدیمی
        try {
            $this->pdo->exec("ALTER TABLE broadcast_state ADD COLUMN total_users INT DEFAULT 0");
        } catch (Exception $e) { }


    }

    public function query($sql, $params = []) {
        try {
            $stmt = $this->pdo->prepare($sql);
            $stmt->execute($params);
            return $stmt;
        } catch (PDOException $e) {
            error_log("Query Error: " . $e->getMessage() . " SQL: $sql");
            return false;
        }
    }
    
    public function lastInsertId() {
        return $this->pdo->lastInsertId();
    }
}
